# Shinydocs Wildcard Index Update
# Shinydocs_Wildcard_Index_Update.ps1
# Version 1.0.1
# Copyright 2022 Shinydocs Corporation
# PowerShell script provided as-is

param(
    [string]$IndexName,
    [string]$IndexUrl,
    [array]$Fields
)

$BaseLocation = Get-Location 

function Log {
    param(
        # Example use:
        # Log -ERR "There was an error"
        [switch]$ERR,
        [switch]$WRN,
        [switch]$INF,
        [switch]$DBG,
        [string]$msg
    )
    $Timestamp = "[{0:MM/dd/yy} {0:HH:mm:ss}]" -f (Get-Date)

    # Create Index Update Logs directory (if it already exists, nothing will happen)
    New-Item -Type Directory -Path "$BaseLocation\Index Update Logs" -Force | Out-Null

    # ERROR
    if ($ERR) {
        $LogLevel = "ERROR:"
        Write-Host "$($Timestamp) $LogLevel $msg" -ForegroundColor Red
        Add-Content "$BaseLocation\Index Update Logs\index-update-task-$(Get-Date -f yyyy-MM-dd).log" ($TimeStamp + ' ' + $LogLevel + ' ' + $msg)
    }
    # Warning
    if ($WRN) {
        $LogLevel = "Warning:"
        Write-Host "$($Timestamp) $LogLevel $msg" -ForegroundColor Yellow
        Add-Content "$BaseLocation\Index Update Logs\index-update-task-$(Get-Date -f yyyy-MM-dd).log" ($TimeStamp + ' ' + $LogLevel + ' ' + $msg)
    }
    # Info
    if ($INF) {
        $LogLevel = "Info:"
        Write-Host "$($Timestamp) $LogLevel $msg" -ForegroundColor White
        Add-Content "$BaseLocation\Index Update Logs\index-update-task-$(Get-Date -f yyyy-MM-dd).log" ($TimeStamp + ' ' + $LogLevel + ' ' + $msg)
    }
    # Debug
    if ($DBG -and $Verbose) {
        $LogLevel = "Debug:"
        Write-Host "$($Timestamp) $LogLevel $msg" -ForegroundColor Magenta
        Add-Content "$BaseLocation\Index Update Logs\index-update-task-$(Get-Date -f yyyy-MM-dd).log" ($TimeStamp + ' ' + $LogLevel + ' ' + $msg)
    }
}

function CloseIndex {
    # Close the index
    try {
        Log -INF "Closing $IndexName. This index will not be available for search or ingestion while closed."
        Invoke-RestMethod -Method Post -Uri "$IndexUrl/$IndexName/_close" -ErrorAction Stop | Out-Null
    }
    catch {
        Log -ERR "$($Error[0])"
        break
    }
}
function OpenIndex {
    # Open the index
    try {
        Log -INF "Opening $IndexName. This index is now available for search or ingestion"
        Invoke-RestMethod -Method Post -Uri "$IndexUrl/$IndexName/_open" -ErrorAction Stop | Out-Null
    }
    catch {
        Log -ERR "$($Error[0])"
        break
    }
    try {
        $done = $false
        while ($done -eq $false) {
            try {
                $IndexStatus = Invoke-RestMethod -Method Get -Uri "$IndexUrl/_cat/indices?v&index=$IndexName&format=json" -ErrorAction Stop
            }
            catch {
                Log -ERR "$($Error[0])"
                Log -DBG "IndexStatus Response: $IndexStatus"
                break
            }
            if ($IndexStatus.status -eq "open") {
                $done = $true
            }
            if ($IndexStatus.status -eq "close") {
                $done = $false
            }
            Start-Sleep 1
        }
    }
    catch {
        Log -ERR "$($Error[0])"
    }
}
function PutNewSettings {
    $WildcardSettings = @"
{
    "settings": {
      "analysis": {
        "tokenizer": {
          "wildcard_tokenizer": {
            "type": "pattern",
            "pattern": "[^A-Za-z0-9_-]",
            "lowercase": true
          }
        },
        "analyzer": {
          "sd_wildcard_analyzer": {
            "tokenizer": "wildcard_tokenizer",
            "filter": [
              "lowercase",
              "shingle"
            ]
          }
        }
      }
    }
  }
"@
    # Post new settings
    try {
        Log -INF "Posting new index settings to $IndexName"
        Invoke-RestMethod -Method Put -Uri "$IndexUrl/$IndexName/_settings" -Body $WildcardSettings -ContentType application/json -ErrorAction Stop | Out-Null
    }
    catch {
        Log -ERR "$($Error[0])"
        break
    }
}
function PutNewMapping {
    param (
        $Field
    )
    # Check the index mapping for the field and if it already has the sd_name_analyzer (which will change which mapping is sent to the index)
    try {
        $AnalyzersGet = Invoke-RestMethod -Method Get -Uri "$IndexUrl/$IndexName/_mapping" -ErrorAction Stop
        $Type = $AnalyzersGet.$IndexName.mappings.psobject.Properties.Name
        if ($AnalyzersGet.$IndexName.mappings.$Type.properties.$Field.analyzer -eq "sd_name_analyzer") {
            $SdAnalyzer = $true
            Log -INF "'sd_name_analyzer' exists in $IndexName for $Field"
        }
    }
    catch {
        Log -ERR "$($Error[0])"
        break
    }

    $WildcardDocMapping = @"
{
    "properties": {
      "$Field": {
        "type": "text",
        "copy_to": ["$($Field + "_enhanced_wildcard")"],
        "fields": {
          "keyword": {
            "type": "keyword",
            "ignore_above": 256
          },
          "$($Field + "_enhanced_wildcard")": {
            "type": "text",
            "analyzer": "sd_wildcard_analyzer"
          }
        }
      }
    }
  }
"@
    $WithSDAnalyzerMapping = @"
{
    "properties": {
      "$Field": {
        "type": "text",
        "analyzer": "sd_name_analyzer",
        "copy_to": "$($Field + "_enhanced_wildcard")",
        "fields": {
          "keyword": {
            "type": "keyword",
            "ignore_above": 256
          },
          "$($Field + "_enhanced_wildcard")": {
            "type": "text",
            "analyzer": "sd_wildcard_analyzer"
          }
        }
      }
    }
  }
"@
    # Post new mapping
    try {
        Log -INF "Post new mapping to $IndexName for field: $Field"
        if ($SdAnalyzer -eq $true) {
            Invoke-RestMethod -Method Put -Uri "$IndexUrl/$IndexName/_mapping/_doc" -Body $WithSDAnalyzerMapping -ContentType application/json -ErrorAction Stop | Out-Null
        }
        else {
            Invoke-RestMethod -Method Put -Uri "$IndexUrl/$IndexName/_mapping/_doc" -Body $WildcardDocMapping -ContentType application/json -ErrorAction Stop | Out-Null
        }
    }
    catch {
        Log -ERR "$($Error[0])"
    }
}
function UpdateDocs {
    # Update docs
    try {
        $DocUpdatePost = Invoke-RestMethod -Method Post -Uri "$IndexUrl/$IndexName/_update_by_query?&wait_for_completion=false" -ErrorAction Stop 
        $TaskId = $DocUpdatePost.task
        Log -INF "Task ID for docs update: $TaskId"
    }
    catch {
        Log -ERR "$($Error[0])"
    }
    return $TaskId
}
function CheckTask {
    param (
        $UpdateId
    )
    try {
        $TaskResponse = Invoke-RestMethod -Method Get -Uri "$IndexUrl/_tasks/$UpdateId" -ErrorAction Stop
    }
    catch {
        Log -ERR "$($Error[0])"
    }
    return $TaskResponse
    
}

CloseIndex

PutNewSettings

OpenIndex

foreach ($Field in $Fields) {
    PutNewMapping $Field
}

$UpdateId = UpdateDocs

$done = $false
while ($done -eq $false) {
    try {
        $TaskStatus = CheckTask $UpdateId
    }
    catch {
        Log -ERR "$($Error[0])"
        break
    }
    if ($TaskStatus.completed -eq $false) {
        $done = $false
        $_updated = $TaskStatus.task.status.updated
        $_total = $TaskStatus.task.status.total
        if ($_updated -gt 0) {
            Write-Progress -Activity "Updating docs in $IndexName" -Status "Progress: $_updated/$_total" -PercentComplete ($_updated / $_total * 100)
        }
                
    }
    if ($TaskStatus.completed -eq $true) {
        $done = $true
        $_updated = $TaskStatus.task.status.updated
        $_total = $TaskStatus.task.status.total
        if ($_updated -gt 0) {
            Write-Progress -Activity "Updating docs in $IndexName" -Status "Progress: $_updated/$_total" -PercentComplete ($_updated / $_total * 100) -Completed
        }
        Log -INF "Index update complete. Updated: $_total doc(s)"
                
    }
    Start-Sleep 1
}